import React from 'react';

// --- REMOTION MOCK SYSTEM (for Canvas preview) ---
const Easing = {
  out: (f) => (t) => f(t),
  in: (f) => (t) => f(1 - t),
  inOut: (f) => (t) => t < 0.5 ? f(2 * t) / 2 : (2 - f(2 * (1 - t))) / 2,
  linear: (t) => t,
  quad: (t) => t * (2 - t),
  cubic: (t) => --t * t * t + 1,
  bezier: (x1, y1, x2, y2) => (t) => {
    const cx = 3 * x1;
    const bx = 3 * (x2 - x1) - cx;
    const ax = 1 - cx - bx;
    const cy = 3 * y1;
    const by = 3 * (y2 - y1) - cy;
    const ay = 1 - cy - by;
    const solveX = (x) => {
      let t2 = x;
      for (let i = 0; i < 8; i++) {
        const x2 = ((ax * t2 + bx) * t2 + cx) * t2 - x;
        if (Math.abs(x2) < 0.001) return t2;
        const d2 = (3 * ax * t2 + 2 * bx) * t2 + cx;
        if (Math.abs(d2) < 1e-6) break;
        t2 -= x2 / d2;
      }
      return t2;
    };
    return ((ay * solveX(t) + by) * solveX(t) + cy) * solveX(t);
  },
  elastic: (bounciness = 1) => (t) => {
    const p = bounciness * Math.PI;
    return 1 - Math.pow(Math.cos((t * Math.PI) / 2), 3) * Math.cos(t * p);
  },
  bounce: (t) => {
    if (t < 1 / 2.75) return 7.5625 * t * t;
    if (t < 2 / 2.75) return 7.5625 * (t -= 1.5 / 2.75) * t + 0.75;
    if (t < 2.5 / 2.75) return 7.5625 * (t -= 2.25 / 2.75) * t + 0.9375;
    return 7.5625 * (t -= 2.625 / 2.75) * t + 0.984375;
  }
};

const interpolate = (input, inputRange, outputRange, options = {}) => {
  const { extrapolateLeft = 'extend', extrapolateRight = 'extend', easing } = options;
  const [minInput, maxInput] = inputRange;
  const [minOutput, maxOutput] = outputRange;

  if (input <= minInput) {
    if (extrapolateLeft === 'clamp') return minOutput;
    return minOutput + (input - minInput) * (maxOutput - minOutput) / (maxInput - minInput);
  }
  if (input >= maxInput) {
    if (extrapolateRight === 'clamp') return maxOutput;
    return minOutput + (input - minInput) * (maxOutput - minOutput) / (maxInput - minInput);
  }

  let progress = (input - minInput) / (maxInput - minInput);
  if (easing) progress = easing(progress);
  return minOutput + progress * (maxOutput - minOutput);
};

const useCurrentFrame = () => {
  const [frame, setFrame] = React.useState(0);
  React.useEffect(() => {
    const interval = setInterval(() => setFrame(f => (f + 1) % 450), 1000 / 30);
    return () => clearInterval(interval);
  }, []);
  return frame;
};

const useVideoConfig = () => ({ 
  width: 1920, 
  height: 1080, 
  fps: 30, 
  durationInFrames: 450 
});

const AbsoluteFill = ({ children, style }) => (
  <div style={{ position: 'absolute', top: 0, left: 0, right: 0, bottom: 0, ...style }}>
    {children}
  </div>
);

// =============================================================================
// PRE-GENERATED DATA
// =============================================================================

const seededRandom = (seed: number): number => {
  const x = Math.sin(seed * 9999) * 10000;
  return x - Math.floor(x);
};

const candleData = [
  { o: 145.5, h: 146.2, l: 144.8, c: 145.9 },
  { o: 145.9, h: 147.1, l: 145.5, c: 146.8 },
  { o: 146.8, h: 147.5, l: 146.2, c: 147.2 },
  { o: 147.2, h: 147.4, l: 145.8, c: 146.1 },
  { o: 146.1, h: 146.9, l: 145.7, c: 146.7 },
  { o: 146.7, h: 148.2, l: 146.5, c: 147.9 },
  { o: 147.9, h: 148.5, l: 147.2, c: 147.5 },
  { o: 147.5, h: 147.8, l: 143.5, c: 144.2 }, 
  { o: 144.2, h: 145.1, l: 142.8, c: 143.5 },
  { o: 143.5, h: 144.2, l: 143.1, c: 144.0 },
  { o: 144.0, h: 146.8, l: 143.8, c: 146.2 }, 
  { o: 146.2, h: 148.5, l: 146.0, c: 148.1 },
  { o: 148.1, h: 149.8, l: 147.8, c: 149.2 },
  { o: 149.2, h: 150.5, l: 149.0, c: 150.1 },
  { o: 150.1, h: 151.2, l: 149.5, c: 150.8 },
];

const priceGrid = [142, 144, 146, 148, 150, 152];

const backgroundParticles = Array.from({ length: 40 }).map((_, i) => ({
  x: seededRandom(i) * 100,
  y: seededRandom(i + 500) * 100,
  size: seededRandom(i + 1000) * 2 + 1,
  opacity: seededRandom(i + 1500) * 0.15 + 0.05,
}));

// =============================================================================
// MAIN COMPONENT
// =============================================================================

const ChartAnimation: React.FC = () => {
  const frame = useCurrentFrame();

  // Scale down more significantly to match the "hero" look of the screenshot
  const chartWidth = 1000; 
  const chartHeight = 450;
  
  const minPrice = 142;
  const maxPrice = 152;
  const priceRange = maxPrice - minPrice;

  const gridOpacity = interpolate(frame, [10, 50], [0, 0.08], {
    extrapolateLeft: 'clamp',
    extrapolateRight: 'clamp',
  });

  return (
    <AbsoluteFill style={{ 
      backgroundColor: '#020617', 
      overflow: 'hidden', 
      fontFamily: 'system-ui, sans-serif',
      display: 'flex',
      justifyContent: 'center',
      alignItems: 'center'
    }}>
      
      {/* Background Ambience */}
      {backgroundParticles.map((p, i) => (
        <div
          key={i}
          style={{
            position: 'absolute',
            left: `${p.x}%`,
            top: `${p.y}%`,
            width: p.size,
            height: p.size,
            borderRadius: '50%',
            backgroundColor: '#38bdf8',
            opacity: p.opacity,
          }}
        />
      ))}

      {/* Center Wrapper: Shifts slightly left to account for right labels */}
      <div style={{
        position: 'relative',
        width: chartWidth,
        height: chartHeight,
        marginRight: 100, // Offset to compensate for right-side axis labels
      }}>
        
        {/* Horizontal Grid Lines & Price Labels on Right */}
        {priceGrid.map((price, i) => {
          const y = chartHeight - ((price - minPrice) / priceRange) * chartHeight;
          const lineReveal = interpolate(frame, [10 + i * 4, 40 + i * 4], [0, 1], { 
            extrapolateLeft: 'clamp', 
            extrapolateRight: 'clamp' 
          });

          return (
            <React.Fragment key={i}>
              <div style={{
                position: 'absolute',
                top: y,
                left: 0,
                width: '100%',
                height: 1,
                backgroundColor: 'rgba(255, 255, 255, 1)',
                opacity: gridOpacity * lineReveal,
              }} />
              <span style={{
                position: 'absolute',
                top: y - 12,
                right: -100,
                fontSize: 22,
                color: '#475569',
                fontWeight: 600,
                opacity: lineReveal,
              }}>
                ${price.toFixed(0)}
              </span>
            </React.Fragment>
          );
        })}

        {/* Candlesticks Rendering */}
        <div style={{ position: 'relative', width: '100%', height: '100%' }}>
          {candleData.map((d, i) => {
            const x = (i / (candleData.length)) * chartWidth + (chartWidth / candleData.length / 2);
            const candleWidth = (chartWidth / candleData.length) * 0.55; 
            
            const yOpen = chartHeight - ((d.o - minPrice) / priceRange) * chartHeight;
            const yClose = chartHeight - ((d.c - minPrice) / priceRange) * chartHeight;
            const yHigh = chartHeight - ((d.h - minPrice) / priceRange) * chartHeight;
            const yLow = chartHeight - ((d.l - minPrice) / priceRange) * chartHeight;
            
            const isGreen = d.c >= d.o;
            const candleColor = isGreen ? '#10b981' : '#ef4444';

            const revealStart = 30 + i * 8;
            const candleReveal = interpolate(frame, [revealStart, revealStart + 15], [0, 1], {
              extrapolateLeft: 'clamp',
              extrapolateRight: 'clamp',
              easing: Easing.out(Easing.cubic),
            });

            const bodyTop = Math.min(yOpen, yClose);
            const bodyHeight = Math.max(Math.abs(yOpen - yClose), 2);

            return (
              <div key={i} style={{ opacity: candleReveal }}>
                {/* Wick */}
                <div style={{
                  position: 'absolute',
                  left: x,
                  top: yHigh,
                  width: 1.5,
                  height: yLow - yHigh,
                  backgroundColor: candleColor,
                  transform: 'translateX(-50%)',
                  opacity: 0.8,
                }} />
                
                {/* Body */}
                <div style={{
                  position: 'absolute',
                  left: x,
                  top: bodyTop,
                  width: candleWidth,
                  height: bodyHeight,
                  backgroundColor: candleColor,
                  transform: `translateX(-50%) scaleY(${candleReveal})`,
                  borderRadius: 1,
                  boxShadow: `0 0 10px ${isGreen ? 'rgba(16, 185, 129, 0.2)' : 'rgba(239, 68, 68, 0.2)'}`,
                }} />
              </div>
            );
          })}
        </div>

        {/* Price Tag (Right Side) */}
        {frame > 160 && (
          <div style={{
            position: 'absolute',
            right: -130,
            top: chartHeight - ((candleData[candleData.length - 1].c - minPrice) / priceRange) * chartHeight - 18,
            backgroundColor: candleData[candleData.length - 1].c >= candleData[candleData.length - 1].o ? '#10b981' : '#ef4444',
            padding: '6px 12px',
            borderRadius: '4px',
            color: 'white',
            fontWeight: 800,
            fontSize: 22,
            boxShadow: '0 4px 10px rgba(0,0,0,0.3)',
            zIndex: 10,
            opacity: interpolate(frame, [160, 180], [0, 1], { extrapolateLeft: 'clamp' }),
          }}>
            ${candleData[candleData.length - 1].c.toFixed(2)}
            <div style={{
              position: 'absolute',
              left: -6,
              top: '50%',
              transform: 'translateY(-50%) rotate(45deg)',
              width: 12,
              height: 12,
              backgroundColor: candleData[candleData.length - 1].c >= candleData[candleData.length - 1].o ? '#10b981' : '#ef4444',
              zIndex: -1
            }} />
          </div>
        )}
      </div>

    </AbsoluteFill>
  );
};

export default ChartAnimation;

export const RemotionVideo = {
  component: ChartAnimation,
  durationInFrames: 450,
  fps: 30,
  width: 1920,
  height: 1080,
  id: 'ChartAnimation',
};